﻿using System;
using System.Threading;
using Microsoft.SPOT;
using Microsoft.SPOT.Hardware;
using SecretLabs.NETMF.Hardware;
using SecretLabs.NETMF.Hardware.Netduino;
using System.Runtime.CompilerServices;
using System.IO;

namespace Mp3Player
{
   public class Vs1053B
   {
      private InputPort MP3_DREQ;
      private OutputPort MP3_RESET;
      private SPI _spi;
      private SPI.Configuration sci_config;
      private SPI.Configuration sdi_config;
      private byte[] cmd_buffer;
      private byte LeftChannelVolume;
      private byte RightChannelVolume;
      const ushort para_endFillByte = 0x1E06;
      private const byte CMD_WRITE = 0x02;
      private const byte CMD_READ = 0x03;
      private const ushort SM_RESET = 0x04;
      private const ushort SM_CANCEL = 0x10;
      private const ushort SM_TESTS = 0x20;
      private const ushort SM_SDINEW = 0x800;
      private const ushort SM_ADPCM = 0x1000;
      private const ushort SM_LINE1 = 0x4000;
      private const int SCI_MODE = 0x00;
      private const int SCI_STATUS = 0x01;
      private const int SCI_BASS = 0x02;
      private const int SCI_CLOCKF = 0x03;
      private const int SCI_WRAM = 0x06;
      private const int SCI_WRAMADDR = 0x07;
      private const int SCI_HDAT0 = 0x08;
      private const int SCI_HDAT1 = 0x09;
      private const int SCI_AIADDR = 0x0A;
      private const int SCI_VOL = 0x0B;
      private const int SCI_AICTRL0 = 0x0C;
      private const int SCI_AICTRL1 = 0x0D;
      private const int SCI_AICTRL2 = 0x0E;
      private const int SCI_AICTRL3 = 0x0F;
      private static bool _busy = false;      // Playing a file when true
      private static bool _cancelMusic = false;
      private static long _readSoFar = -1;
      private static int _lastPos = -1;
      private static long _rawSize = -1;
      private static bool _pause = false;
      public static object _source;
      private static byte[] block = new byte[32];
      private static byte[] cmdBuffer = new byte[4];

      private bool cancelPlayback = false;
      public event musicFinished onMusicFinished;
      public delegate void musicFinished();

      private ushort mp3_sci_read(byte register)
      {
         ushort temp;
         _spi.Config = sci_config;
         while (!MP3_DREQ.Read()) ;
         cmd_buffer[0] = CMD_READ;
         cmd_buffer[1] = register;
         cmd_buffer[2] = 0;
         cmd_buffer[3] = 0;
         byte[] readBuffer = new byte[4];
         _spi.WriteRead(cmd_buffer, readBuffer);
         temp = readBuffer[2];
         temp <<= 8;
         temp += readBuffer[3];
         return temp;
      }
      private void mp3_sci_write(byte register, ushort data)
      {
         _spi.Config = sci_config;
         while (!MP3_DREQ.Read()) ;
         cmd_buffer[0] = CMD_WRITE;
         cmd_buffer[1] = register;
         cmd_buffer[2] = (byte)(data >> 8);
         cmd_buffer[3] = (byte)data;
         _spi.Write(cmd_buffer);
      }
      private void Reset()
      {
         while (!MP3_DREQ.Read()) ;
         mp3_sci_write(SCI_MODE, SM_SDINEW | SM_RESET);
         while (!MP3_DREQ.Read()) ;
         mp3_sci_write(SCI_CLOCKF, 0xa000);
      }

      public Vs1053B(Cpu.Pin MP3_DREQ, Cpu.Pin MP3_CS, Cpu.Pin MP3_DCS, Cpu.Pin MP3_RST)
      {
         cmd_buffer = new byte[4];
         this.sci_config = new SPI.Configuration(MP3_CS, false, 0, 0, false, true, 1000, SPI_Devices.SPI1);
         this.sdi_config = new SPI.Configuration(MP3_DCS, false, 0, 0, false, true, 1000, SPI_Devices.SPI1);
         this.MP3_DREQ = new InputPort(MP3_DREQ, false, Port.ResistorMode.Disabled);
         this.MP3_RESET = new OutputPort(MP3_RST, true);
         this._spi = new SPI(sci_config);
         MP3_RESET.Write(false);
         Thread.Sleep(1);
         MP3_RESET.Write(true);
         Thread.Sleep(10);
         Reset();
         mp3_sci_write(SCI_MODE, SM_SDINEW);
         mp3_sci_write(SCI_CLOCKF, 0x8800);
         Thread.Sleep(100);
         SPI.Configuration temp = sdi_config;
         sdi_config = new SPI.Configuration(temp.ChipSelect_Port, temp.BusyPin_ActiveState, temp.ChipSelect_SetupTime, temp.ChipSelect_HoldTime, temp.Clock_IdleState, temp.Clock_Edge, 3000, temp.SPI_mod);
         Thread.Sleep(100);
         mp3_sci_write(SCI_VOL, 0x0101);
         if (mp3_sci_read(SCI_VOL) != (0x0101))
         {
            throw new Exception("Initialization Error");
         }
         SetVolume(200, 200);
      }
      public void SetVolume(byte LeftVolume, byte RightVolume)
      {
         if (LeftVolume > 255)
            LeftChannelVolume = 255;
         else if (LeftVolume < 0)
            LeftChannelVolume = 0;
         else
            LeftChannelVolume = LeftVolume;

         if (RightVolume > 255)
            RightChannelVolume = 255;
         else if (RightVolume < 0)
            RightChannelVolume = 0;
         else
            RightChannelVolume = RightVolume;

         mp3_sci_write(SCI_VOL, (ushort)((255 - LeftVolume) << 8 | (255 - RightVolume)));
      }

      [MethodImpl(MethodImplOptions.Synchronized)]
      public void SetVolume(byte volume)
      {
         if (volume > 255)
            LeftChannelVolume = RightChannelVolume = 255;
         else if (volume < 0)
            LeftChannelVolume = RightChannelVolume = 0;
         else
            LeftChannelVolume = RightChannelVolume = volume;

         mp3_sci_write(SCI_VOL, (ushort)((255 - LeftChannelVolume) << 8 | (255 - RightChannelVolume)));
      }

      public void Play(byte[] data, bool ResetWhenFinished = false)
      {
         if (_spi.Config != sdi_config)
            _spi.Config = sdi_config;
         byte[] block = new byte[32];
         int size = data.Length - data.Length % 32;
         int left_over = data.Length - size;
         for (int i = 0; i < size; i += 32)
         {
            Array.Copy(data, i, block, 0, 32);
            while (!MP3_DREQ.Read()) ;
            _spi.Write(block);
         }
         block = null;
         block = new byte[left_over];
         Array.Copy(data, size, block, 0, left_over);
         while (!MP3_DREQ.Read()) ;
         _spi.Write(block);
         if (ResetWhenFinished)
            Reset();
      }

      public void Play(string FileName, bool ResetWhenFinished = false)
      {
         byte[] block = new byte[32];
         using (FileStream stream = new FileStream(FileName, FileMode.Open, FileAccess.Read))
         {
            long size = stream.Length - stream.Length % 32;
            int left_over = (int)(stream.Length - size);
            for (int i = 0; i < size; i += 32)
            {
               stream.Read(block, 0, 32);
               while (!MP3_DREQ.Read()) ;
               if (cancelPlayback)
               {
                  break;
               }
               SendData(block);
            }
            if (!cancelPlayback)
            {
               block = null;
               block = new byte[left_over];
               stream.Read(block, 0, left_over);
               while (!MP3_DREQ.Read()) ;
               SendData(block);
            }

         }
         if (ResetWhenFinished) // || cancelPlayback)
         {
            Reset();
         }

         if (cancelPlayback)
         {
            cancelPlayback = false;
         }
      }

      public void PlayLoop()
      {
         while (true)
         {
            byte[] block = new byte[32];

            using (FileStream stream = new FileStream((string)_source, FileMode.Open, FileAccess.Read))
            {
               long size = stream.Length - stream.Length % 32;
               int left_over = (int)(stream.Length - size);
               for (int i = 0; i < size; i += 32)
               {
                  if (cancelPlayback)
                  {
                     break;
                  }

                  stream.Read(block, 0, 32);
                  while (!MP3_DREQ.Read()) ;
                  SendData(block);
               }
               if (!cancelPlayback)
               {
                  block = null;
                  block = new byte[left_over];
                  stream.Read(block, 0, left_over);
                  while (!MP3_DREQ.Read()) ;
                  SendData(block);

                  if (onMusicFinished != null)
                  {
                     onMusicFinished();
                  }
               }
               if (cancelPlayback)
               {
                  cancelPlayback = false;
               }
            }

         }
      }

      public string Filename
      {
         get
         {
            if (_source is string)
               return (string)_source;
            else
               return "Embedded Resource";
         }
         set { _source = value; }
      }

      [MethodImpl(MethodImplOptions.Synchronized)]
      public void Play()
      {
         new Thread(PlayLoop).Start();
      }

      [MethodImpl(MethodImplOptions.Synchronized)]
      public void SendData(byte[] data)
      {
         int size = data.Length - data.Length % 32;

         //if (cancelPlayback)
         //{
         //   Debug.Print("Wrote SM_CANCEL");
         //   SCIWrite(SCI_MODE, SM_CANCEL);
         //}

         ushort CANCEL = 0;
         for (int i = 0; i < size; i += 32)
         {
            Array.Copy(data, i, block, 0, 32);
            SDIWrite(block);
            //Debug.Print("Wrote 32 bytes of data");
            if (cancelPlayback)
            {
               break;
               //CANCEL = SCIRead(SCI_MODE);
               //Debug.Print("Checking if SM_CANCEL has cleared (" + CANCEL + ")");
               //if (CANCEL != SM_CANCEL)
               //{
               //   Debug.Print("SM_CANCEL cleared");
               //   StopPlayback();
               //   break;
               //}
               //else
               //   Debug.Print("SM_CANCEL has not cleared yet...");
            }
            while (_pause)
            {
               Thread.Sleep(1);
            }
         }
         //if (cancelPlayback)
         //{
         //   StopPlayback();
         //   cancelPlayback = false;
         //}
         //if (CANCEL == SM_CANCEL)
         //{
         //    Reset();
         //    cancelPlayback = false;
         //}
      }

      public void CancelPlayback()
      {
         cancelPlayback = true;
      }

      public bool IsPaused()
      {
         return _pause;
      }

      public bool IsCancelling()
      {
         return cancelPlayback;
      }

      public void Pause()
      {
         _pause = true;
      }

      public void Resume()
      {
         _pause = false;
      }

      public void StopPlayback()
      {
         cancelPlayback = false;
         Debug.Print("Stopping playback");
         uint endFillByte = WRAMRead(para_endFillByte);
         Debug.Print("Read endFillByte: " + endFillByte);
         ushort HDAT0;
         ushort HDAT1;
         do
         {
            for (int n = 0; n < 2052; n++) SDIWrite((byte)(0xFF & endFillByte));
            Debug.Print("Sent 2052 endFillByte, checking HDAT0 and HDAT1");
            HDAT0 = SCIRead(SCI_HDAT0);
            HDAT1 = SCIRead(SCI_HDAT1);
            Debug.Print("HDAT0: " + HDAT0 + ", HDAT1: " + HDAT1);
         }
         while (HDAT0 != 0 && HDAT1 != 0);
      }



      /// <summary>
      /// Method from reading from WRAM.
      /// </summary>
      /// <param name="address"></param>
      /// <returns></returns>
      private ushort WRAMRead(ushort address)
      {
         ushort tmp1, tmp2;
         SCIWrite(SCI_WRAMADDR, address);
         tmp1 = SCIRead(SCI_WRAM);
         SCIWrite(SCI_WRAMADDR, address);
         tmp2 = SCIRead(SCI_WRAM);
         if (tmp1 == tmp2) return tmp1;
         SCIWrite(SCI_WRAMADDR, address);
         tmp1 = SCIRead(SCI_WRAM);
         if (tmp1 == tmp2) return tmp1;
         SCIWrite(SCI_WRAMADDR, address);
         tmp1 = SCIRead(SCI_WRAM);
         if (tmp1 == tmp2) return tmp1;
         return tmp1;
      }

      /// <summary>
      /// Metod for setting the volume on the left and right channel.
      /// <remarks>0 for min volume, 255 for max</remarks>
      /// </summary>
      /// <param name="left_channel">The left channel.</param>
      /// <param name="right_channel">The right channel.</param>
      private void SetVolume1(byte volume)
      {
         ushort vol = (ushort)(((255 - volume) << 8) | (255 - volume));
         while (!MP3_DREQ.Read())
            Thread.Sleep(1);
         SCIWrite(SCI_VOL, vol);
      }

      /// <summary>
      /// Method for writing data to the decoder.
      /// </summary>
      /// <param name="writeBytes">Bytes to write.</param>
      private void SDIWrite(byte[] writeBytes)
      {
         if (_spi.Config != sdi_config)
            _spi.Config = sdi_config;
         while (!MP3_DREQ.Read())
            Thread.Sleep(1);
         _spi.Write(writeBytes);
      }

      /// <summary>
      /// Method for writing a single byte to the decoder.
      /// </summary>
      /// <param name="writeByte">Byte to write.</param>
      private void SDIWrite(byte writeByte)
      {
         SDIWrite(new byte[] { writeByte });
      }

      /// <summary>
      /// Method for writing a command to the decoder.
      /// </summary>
      /// <param name="address">The address to write to.</param>
      /// <param name="data">The data to write.</param>
      private void SCIWrite(byte address, ushort data)
      {
         while (!MP3_DREQ.Read())
            Thread.Sleep(1);

         _spi.Config = sci_config;
         cmdBuffer[0] = 0x02;
         cmdBuffer[1] = address;
         cmdBuffer[2] = (byte)(data >> 8);
         cmdBuffer[3] = (byte)data;

         _spi.Write(cmdBuffer);
      }

      /// <summary>
      /// Method for reading a command from the decoder.
      /// </summary>
      /// <param name="address">The address to read from.</param>
      /// <returns>The data read.</returns>
      private ushort SCIRead(byte address)
      {
         ushort temp;

         while (!MP3_DREQ.Read())
            Thread.Sleep(1);

         _spi.Config = sci_config;
         cmdBuffer[0] = 0x03;
         cmdBuffer[1] = address;
         cmdBuffer[2] = 0;
         cmdBuffer[3] = 0;

         _spi.WriteRead(cmdBuffer, cmdBuffer, 2);

         temp = cmdBuffer[0];
         temp <<= 8;

         temp += cmdBuffer[1];

         return temp;
      }
   }
}