//---------------------------------------------------------------------------------
// Copyright (c) 2015, devMobile Software
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//---------------------------------------------------------------------------------
using System;
using System.Net;
using System.IO;
using System.Text;
using System.Threading;
using Microsoft.SPOT;
using Microsoft.SPOT.Hardware;
using Microsoft.SPOT.Net.NetworkInformation;
using SecretLabs.NETMF.Hardware.Netduino;

using Amqp;
using Amqp.Framing;
using devMobile.Utilities;
using Gralin.NETMF.Nordic;
using Microsoft.ServiceBus.Micro;



namespace devMobile.IoT.Netduino
{
   class EventHubGateway
   {
      private readonly NRF24L01Plus nRF2L01Plus;
      private readonly OutputPort activityLed = new OutputPort(Pins.ONBOARD_LED, false);
      private readonly ConfigurationManager appSettings = new ConfigurationManager("\\sd", "app.config");
      private System.Object lockThis = new System.Object();
      private Timer uploader;

      // LAN Settings
      private const string ntpServerHostnameSetting = "ntpServerHostname";

      // nRF2L01 settings
      private const string nRF2L01ChannelSetting = "nRF2L01Channel";
      private const string nRF2L01AddressSetting = "nRF2L01Address";
      private const string nRF2L01DataRateSetting = "nRF2L01DataRate";

      // Azure ServiceBus Event Hub Settings
      private const string serviceBusHostSetting = "serviceBusHost"; 
      private string serviceBusHost;
      private const string serviceBusPortSetting = "serviceBusPort"; 
      private int serviceBusPort;
      private const string serviceBusSasKeyNameSetting = "serviceBusSasKeyName"; 
      private string serviceBusSasKeyName;
      private const string serviceBusSasKeySetting = "serviceBusSasKey"; 
      private string serviceBusSasKey;
      private const string eventHubNameSetting = "serviceBusEventHubName";
      private string eventHubName ;
      private string deviceId;
      private const string gatewayIdSetting = "gatewayId";
      private string gatewayId;
      private const string eventUploadBatchSizeMaximumSetting = "eventUploadBatchSizeMaximum";
      private int eventUploadBatchSizeMaximum;
      private const string eventUploadBatchPeriodSetting = "eventUploadBatchPeriod ";
      private int eventUploadBatchPeriod;

      
      
      public EventHubGateway()
      {
         nRF2L01Plus = new NRF24L01Plus();
      }

      
      
      public void Setup()
      {
         // Write empty template of the configuration settings to the SD card if pin D0 is high
         if (!File.Exists(Path.Combine("\\sd", "app.config")))
         {
            Debug.Print("Writing template configuration file then stopping");

            ConfigurationFileGenerate();

            Thread.Sleep(Timeout.Infinite);
         }

         appSettings.Load();

         // Create the queue directory
         if (!Directory.Exists( Path.Combine("\\sd", "Data")))
         {
            Directory.CreateDirectory(Path.Combine("\\sd", "Data"));
         }

         if (!Directory.Exists(Path.Combine("\\sd", "Temp")))
         {
            Directory.CreateDirectory(Path.Combine("\\sd", "Temp"));
         }

         // Wait for Network address if DHCP
         NetworkInterface networkInterface = NetworkInterface.GetAllNetworkInterfaces()[0];
         if (networkInterface.IsDhcpEnabled)
         {
            Debug.Print(" Waiting for DHCP IP address");

            while (NetworkInterface.GetAllNetworkInterfaces()[0].IPAddress == IPAddress.Any.ToString())
            {
               Debug.Print(" .");
               Thread.Sleep(250);
            }
         }

         // Display network config for debugging
         Debug.Print("Network configuration");
         Debug.Print(" Network interface type : " + networkInterface.NetworkInterfaceType.ToString());
         Debug.Print(" MAC Address : " + BytesToHexString(networkInterface.PhysicalAddress));
         Debug.Print(" DHCP enabled : " + networkInterface.IsDhcpEnabled.ToString());
         Debug.Print(" Dynamic DNS enabled : " + networkInterface.IsDynamicDnsEnabled.ToString());
         Debug.Print(" IP Address : " + networkInterface.IPAddress.ToString());
         Debug.Print(" Subnet Mask : " + networkInterface.SubnetMask.ToString());
         Debug.Print(" Gateway : " + networkInterface.GatewayAddress.ToString());

         foreach (string dnsAddress in networkInterface.DnsAddresses)
         {
            Debug.Print(" DNS Server : " + dnsAddress.ToString());
         }

         NetworkChange.NetworkAvailabilityChanged += NetworkChange_NetworkAvailabilityChanged;
         NetworkChange.NetworkAddressChanged += NetworkChange_NetworkAddressChanged;
      
         // Load settings for the NTP server connection
         Debug.Print("LAN settings");
         string ntpServerHostname = appSettings.GetString(ntpServerHostnameSetting, "time.windows.com");
         Debug.Print(" NTP Server Hostname :" + ntpServerHostname);
         
         // Load settings for Azure Event Hub
         Debug.Print("Azure ServiceBus settings");
         serviceBusHost = appSettings.GetString(serviceBusHostSetting); 
         Debug.Print(" ServiceBus Host :" + serviceBusHost);

         serviceBusPort = appSettings.GetInteger(serviceBusPortSetting);
         Debug.Print(" ServiceBus Port :" + serviceBusPort.ToString());

         serviceBusSasKeyName = appSettings.GetString(serviceBusSasKeyNameSetting);
         Debug.Print(" ServiceBus Sas Key name :" + serviceBusSasKeyName);

         serviceBusSasKey = appSettings.GetString(serviceBusSasKeySetting);
         Debug.Print(" ServiceBus Sas Key :" + serviceBusSasKey);

         eventHubName = appSettings.GetString(eventHubNameSetting);
         Debug.Print(" Event Hub name :" + eventHubName);


         // Device settings gateway settings
         Debug.Print("Device settings");
         gatewayId = appSettings.GetString(gatewayIdSetting);
         Debug.Print(" Gateway ID :" + gatewayId);

         deviceId = BytesToHexString(networkInterface.PhysicalAddress);
         Debug.Print(" Device ID :" + deviceId);

         eventUploadBatchSizeMaximum = appSettings.GetInteger(eventUploadBatchSizeMaximumSetting);
         Debug.Print(" Event upload batch size maximum :" + eventUploadBatchSizeMaximum);

         eventUploadBatchPeriod = appSettings.GetInteger(eventUploadBatchPeriodSetting);
         Debug.Print(" Event batch upload period :" + eventUploadBatchPeriod);


         // Load settings for the nRF24l01 wireless
         Debug.Print("nRF24l01 settings");
         string nRF2L01Address = appSettings.GetString(nRF2L01AddressSetting);
         Debug.Print(" Address :" + nRF2L01Address);

         byte nRF24l01Channel = appSettings.GetByte(nRF2L01ChannelSetting);
         Debug.Print(" Channel :" + nRF24l01Channel.ToString());

         NRFDataRate nRF2L01DataRate = (NRFDataRate)appSettings.GetInteger(nRF2L01DataRateSetting);
         Debug.Print(" DataRate :" + nRF2L01DataRate.ToString());


         // Get and set the device time
         Debug.Print("Network time");
         try
         {
            DateTime networkTime = NtpClient.GetNetworkTime(ntpServerHostname);
            Microsoft.SPOT.Hardware.Utility.SetLocalTime(networkTime);
            Debug.Print(networkTime.ToString(" dd-MM-yy HH:mm:ss"));
         }
         catch (Exception ex)
         {
            Debug.Print("ERROR: NtpClient.GetNetworkTime: " + ex.Message);
            Thread.Sleep(Timeout.Infinite);
         }
         Debug.Print("Network time done");


         // Startup the nRF2L01Plus 
         Debug.Print("nRF2L01Plus setup");
         try
         {
            Debug.Print("nRF2L01Plus configuration");
            nRF2L01Plus.OnDataReceived += OnReceive;
            nRF2L01Plus.OnTransmitFailed += OnSendFailure;
            nRF2L01Plus.OnTransmitSuccess += OnSendSuccess;
            nRF2L01Plus.Initialize(SPI.SPI_module.SPI1, Pins.GPIO_PIN_D7, Pins.GPIO_PIN_D3, Pins.GPIO_PIN_D2);
            nRF2L01Plus.Configure(UTF8Encoding.UTF8.GetBytes(nRF2L01Address), nRF24l01Channel, nRF2L01DataRate);

            nRF2L01Plus.Enable();
         }
         catch (Exception ex)
         {
            Debug.Print("ERROR: nRF2L01Plus configuration: " + ex.Message);
            Thread.Sleep(Timeout.Infinite);
         }
         Debug.Print("nRF2L01Plus setup done");

         Debug.Print("Upload Timer");
         uploader = new Timer(uploaderCallback, null, eventUploadBatchPeriod, eventUploadBatchPeriod);

         Debug.Print("Application startup completed");
      }



      void NetworkChange_NetworkAvailabilityChanged(object sender, NetworkAvailabilityEventArgs e)
      {
         Debug.Print(" NetworkChange_NetworkAvailabilityChanged " + e.IsAvailable );

         if (e.IsAvailable)
         {
            uploader = new Timer(uploaderCallback, null, eventUploadBatchPeriod, eventUploadBatchPeriod);
         }
         else
         {
            uploader = new Timer(uploaderCallback, null, Timeout.Infinite, Timeout.Infinite);
         }
      }

      

      void NetworkChange_NetworkAddressChanged(object sender, EventArgs e)
      {
         NetworkInterface networkInterface = NetworkInterface.GetAllNetworkInterfaces()[0];

         Debug.Print("NetworkChange_NetworkAddressChanged " + networkInterface.IPAddress );
      }

      
      
      public void Loop()
      {
         Thread.Sleep(200);
      }

      
      
      private void OnSendSuccess()
      {
         Debug.Print("Send Success!");
      }



      private void OnSendFailure()
      {
         Debug.Print("Send failed!");
      }

      
      
      private void OnReceive(byte[] data)
      {
         activityLed.Write(!activityLed.Read());

         // Ensure that we have a payload
         if (data.Length < 1 )
         {
            Debug.Print( "ERROR - Message has no payload" ) ;
            return ;
         }

         string message = new String(Encoding.UTF8.GetChars(data));
         Debug.Print("+" + DateTime.UtcNow.ToString("HH:mm:ss") + " L=" + data.Length + " M=" + message);

         string filename = DateTime.UtcNow.ToString("yyyyMMddhhmmssff") + ".txt";

         string tempDirectory = Path.Combine("\\sd", "temp");
         string tempFilePath = Path.Combine(tempDirectory, filename);

         string queueDirectory = Path.Combine("\\sd", "data");
         string queueFilePath = Path.Combine(queueDirectory, filename);

         File.WriteAllBytes(tempFilePath, data);

         File.Move(tempFilePath, queueFilePath);

         new Microsoft.SPOT.IO.VolumeInfo("\\sd").FlushAll();
      }


      bool UploadInProgress = false;

      
      void uploaderCallback(object state)
      {
         Debug.Print("uploaderCallback - start");

         if (UploadInProgress)
         {
            return;
         }
         UploadInProgress = true;


         string[] eventFilesToSend = Directory.GetFiles(Path.Combine("\\sd", "data")) ;

         if ( eventFilesToSend.Length == 0 )
         {
            Debug.Print("uploaderCallback - no files");
            UploadInProgress = false;
            return ;
         }

         try
         {
            Debug.Print("uploaderCallback - Connect");
            Connection connection = new Connection(new Address(serviceBusHost, serviceBusPort, serviceBusSasKeyName, serviceBusSasKey));

            Session session = new Session(connection);

            SenderLink sender = new SenderLink(session, "send-link", eventHubName);

            for (int index = 0; index < System.Math.Min(eventUploadBatchSizeMaximum, eventFilesToSend.Length); index++)
            {
               string eventFile = eventFilesToSend[ index ] ;

               Debug.Print("-" + DateTime.UtcNow.ToString("HH:mm:ss") + " " + eventFile ); ;

               Message message = new Message()
               {
                  BodySection = new Data()
                  {
                     Binary = File.ReadAllBytes(eventFile),
                  },
                  ApplicationProperties = new Amqp.Framing.ApplicationProperties(),
               };

               FileInfo fileInfo = new FileInfo(eventFile);

               message.ApplicationProperties["AcquiredAtUtc"] = fileInfo.CreationTimeUtc;
               message.ApplicationProperties["UploadedAtUtc"] = DateTime.UtcNow;
               message.ApplicationProperties["GatewayId"] = gatewayId;
               message.ApplicationProperties["DeviceId"] = deviceId;
               message.ApplicationProperties["EventId"] = Guid.NewGuid();

               sender.Send(message);

               File.Delete(eventFile);

               new Microsoft.SPOT.IO.VolumeInfo("\\sd").FlushAll();
            }

            sender.Close();
            session.Close();
            connection.Close();
         }
         catch (Exception ex)
         {
            Debug.Print("ERROR: Upload failed with error: " + ex.Message);
         }
         finally
         {
            Debug.Print("uploaderCallback - finally");
            UploadInProgress = false;
         }
      }

      
      
      private void ConfigurationFileGenerate()
      {
         // Write empty configuration file
         appSettings.SetString(nRF2L01AddressSetting, "Base1");
         appSettings.SetString(nRF2L01ChannelSetting, "10");
         appSettings.SetString(nRF2L01DataRateSetting, "0");

         appSettings.SetString(eventUploadBatchSizeMaximumSetting, "10");
         appSettings.SetString(eventUploadBatchPeriodSetting, "30000");

         appSettings.SetString(serviceBusHostSetting, "serviceBusHost");
         appSettings.SetString(serviceBusPortSetting, "5671");
         appSettings.SetString(serviceBusSasKeyNameSetting, "serviceBusSasKeyName");
         appSettings.SetString(serviceBusSasKeySetting, "serviceBusSasKey");
         appSettings.SetString(eventHubNameSetting, "eventHubName");

         appSettings.Save();
      }

      
      
      private string BytesToHexString(byte[] bytes)
      {
         string hexString = string.Empty;

         // Create a character array for hexidecimal conversion.
         const string hexChars = "0123456789ABCDEF";

         // Loop through the bytes.
         for (byte b = 0; b < bytes.Length; b++)
         {
            if (b > 0)
               hexString += "-";

            // Grab the top 4 bits and append the hex equivalent to the return string.        
            hexString += hexChars[bytes[b] >> 4];

            // Mask off the upper 4 bits to get the rest of it.
            hexString += hexChars[bytes[b] & 0x0F];
         }

         return hexString;
      }
   }
}