//---------------------------------------------------------------------------------
// Copyright (c) 2015, devMobile Software
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//---------------------------------------------------------------------------------
using System;
using System.Collections;
using System.IO;
using Microsoft.SPOT;



namespace devMobile.Utilities
{    
   public class ConfigurationManager    
   {
      private string configurationFileVolume = "";
      private string configurationFilename = "";
      private Hashtable settings ;
      private const char splitter = '=';
     
      
      public ConfigurationManager(string configVolume, string configFilename)        
      {            
         #region Assertions
         Debug.Assert(configVolume != null);
         Debug.Assert(configVolume.Length > 0);
         Debug.Assert(configFilename != null);
         Debug.Assert(configFilename.Length > 0);
         #endregion

         settings = new Hashtable();
         configurationFileVolume = configVolume;
         configurationFilename = Path.Combine(configVolume, configFilename); 
      }

      
      
      private string prepareKey(string key)
      {
         #region Assertions
         Debug.Assert(key != null);
         Debug.Assert(key.Trim().Length > 0);
         Debug.Assert(key.IndexOf(splitter) == -1);
         #endregion

         return key.Trim().ToLower();
      }        

      

      public void SetString(string key, string value)
      {
         #region Assertions
         Debug.Assert(value != null);
         Debug.Assert(value.Length >= 0);
         #endregion

         key = prepareKey(key);

         if (settings.Contains(key))
         {
            settings[key] = value;
         }
         else
         {
            settings.Add(key, value);
         }
      }

      
      
      public string GetString(string key)
      {
         key = prepareKey(key);

         if (!settings.Contains(key))
         {
            throw new ArgumentException("Key contains unknown value " + key, "key");
         }

         return (string)settings[ key ];
      }



      public string GetString(string key, string defaultValue)
      {
         #region Assertions
         Debug.Assert(key != null);
         Debug.Assert(key.Trim().Length > 0);
         Debug.Assert(defaultValue != null);
         Debug.Assert(defaultValue.Length >= 0);
         #endregion

         key = prepareKey(key);

         if (settings.Contains(key))
         {
            return (string)settings[key];
         }

         return defaultValue;
      }



      public byte GetByte(string key)
      {
         #region Assertions
         Debug.Assert(key != null);
         Debug.Assert(key.Trim().Length > 0);
         #endregion

         key = prepareKey(key);

         if (!settings.Contains(key))
         {
            throw new ArgumentException("Key contains unknown value " + key, "key");
         }

         string value = (string)settings[ key ];

         return( byte.Parse(value) );
      }

      
      
      public int GetInteger(string key)
      {
         #region Assertions
         Debug.Assert(key != null);
         Debug.Assert(key.Trim().Length > 0);
         #endregion

         key = prepareKey(key);

         if (!settings.Contains(key))
         {
            throw new ArgumentException("Key contains unknown value " + key, "key");
         }

         string value = (string) settings[ key ] ;

         return( int.Parse( value )) ;
      }

      
      
      public int GetInteger(string key, int defaultValue )
      {
         #region Assertions
         Debug.Assert(key != null);
         Debug.Assert(key.Trim().Length > 0);
         #endregion

         key = prepareKey(key);

         if (settings.Contains(key))
         {
            string value = (string)settings[ key ];

            return (int.Parse(value));
         }

         return defaultValue;
      }



      public void Load()
      {
         settings.Clear();

         if (!File.Exists(configurationFilename))
         {
            return;
         }

         using (var file = new StreamReader(configurationFilename))
         {
            string line;
            while ((line = file.ReadLine()) != null)
            {
               string key = line.Substring( 0, line.IndexOf( splitter ));
               string value = line.Substring(line.IndexOf(splitter) + 1 );

               settings.Add(prepareKey(key), value);
            }
         }
      }       

      
      
      public void Save()        
      {  
         using (var file = new StreamWriter(configurationFilename, false))            
         {                
            foreach (DictionaryEntry dataElement in settings)                
            {                    
               string line = dataElement.Key.ToString() + splitter + dataElement.Value.ToString();                    
               file.WriteLine(line);
            }
            file.Flush();
         }
         new Microsoft.SPOT.IO.VolumeInfo(configurationFileVolume).FlushAll();
      }



      public void Dump()
      {
         Debug.Print("");
         if (settings != null)
         {
            Debug.Print("Dump-Hashtable  contents");
            foreach (DictionaryEntry dataElement in settings)
            {
               Debug.Print( " " + dataElement.Key.ToString() + splitter + dataElement.Value.ToString());
            }
         }

         if (File.Exists(configurationFilename))
         {
            Debug.Print("Dump-File contents");
            using (var file = new StreamReader(configurationFilename))
            {
               string line;
               while ((line = file.ReadLine()) != null)
               {
                  Debug.Print( " " + line); 
               }
            }
         }
      }
   }
}