/*
Using a display like http://www.dfrobot.com/index.php?route=product/product&product_id=51#.U0D-FBuKBhE
 
(C) April 2014 devMobile.co.nz and CodeClub.org.nz
 
Inspired by 
 
http://forums.netduino.com/index.php?/topic/284-lcdkeypad-16x2-hd44780-shield/

*/
using System;
using System.Threading;
using Microsoft.SPOT.Hardware;
using SecretLabs.NETMF.Hardware.Netduino;



namespace CodeClub.Hardware.Display
{
   public class DFRobotLcd16x2
   {
      private OutputPort RegisterSelect;
      private OutputPort Enable;

      private OutputPort DB4;
      private OutputPort DB5;
      private OutputPort DB6;
      private OutputPort DB7;

      private PWM BackLight;

      private const byte CursorSetPosition = 0x80; //SET_CURSOR + X : Sets cursor position to X        

      private const byte DisplayClear = 1;       // Clear display
      private const byte DISPLAY_BLANK = 0x08;    // Blank display without clearing it
      private const byte DISPLAY_RESTORE = 0x0c;  // restore display, cursor hidden



      public DFRobotLcd16x2()
      {
         DB4 = new OutputPort(Pins.GPIO_PIN_D4, false);
         DB5 = new OutputPort(Pins.GPIO_PIN_D5, false);
         DB6 = new OutputPort(Pins.GPIO_PIN_D6, false);
         DB7 = new OutputPort(Pins.GPIO_PIN_D7, false);

         RegisterSelect = new OutputPort(Pins.GPIO_PIN_D8, false);
         Enable = new OutputPort(Pins.GPIO_PIN_D9, false);

         BackLight = new PWM(PWMChannels.PWM_PIN_D10, 100, 0.0, false);
         BackLight.Start();

         RegisterSelect.Write(false);

         // 4 bit data communication
         Thread.Sleep(50);

         DB7.Write(false);
         DB6.Write(false);
         DB5.Write(true);
         DB4.Write(true);

         Enable.Write(true);
         Enable.Write(false);

         Thread.Sleep(50);
         DB7.Write(false);
         DB6.Write(false);
         DB5.Write(true);
         DB4.Write(true);

         Enable.Write(true);
         Enable.Write(false);

         Thread.Sleep(50);
         DB7.Write(false);
         DB6.Write(false);
         DB5.Write(true);
         DB4.Write(true);

         Enable.Write(true);
         Enable.Write(false);

         Thread.Sleep(50);
         DB7.Write(false);
         DB6.Write(false);
         DB5.Write(true);
         DB4.Write(false);

         Enable.Write(true);
         Enable.Write(false);

         SendCmd(DISPLAY_RESTORE);
         SendCmd(DisplayClear);
      }



      /// <summary>
      /// Places the cursor at a given row and column
      /// Top left is 0 , 0
      /// </summary>
      /// <param name="row"></param>
      /// <param name="col"></param>
      public void CursorSet(byte row, byte col)
      {
         SendCmd((byte)(CursorSetPosition | row << 6 | col));
      }



      /// <summary>
      /// Remove all data from all lines of the display
      /// </summary>
      public void Clear(int delayBefore = 0)
      {
         Thread.Sleep(delayBefore);
         SendCmd(DisplayClear);
      }



      /// <summary>
      /// Blank the display without clearing it
      /// </summary>
      public void DisplayHide()
      {
         SendCmd(DISPLAY_BLANK);
      }



      /// <summary>
      /// Restore display, cursor hidden
      /// </summary>
      public void DisplayRestore()
      {
         SendCmd(DISPLAY_RESTORE);
      }



      public void BacklightOn()
      {
         BackLight.DutyCycle = 1.0;
      }


      public void BacklightOff()
      {
         BackLight.DutyCycle = 0.0;
      }


      public double BacklightBrightness
      {
         get { return BackLight.DutyCycle; }
         set { BackLight.DutyCycle = value; }
      }


      /// <summary>
      /// Print a string at current cursor location  
      /// </summary>
      /// <param name="str"></param>      
      public void Print(string str)
      {
         for (int i = 0; i < str.Length; i++)
         {
            Putc((byte)str[i]);
         }
      }



      /// <summary>
      /// Print a string at a location  
      /// </summary>
      /// <param name="row"></param>
      /// <param name="col"></param>
      /// <param name="str"></param>
      public void Print(byte row, byte col, string str)
      {
         CursorSet(row, col);
         Print(str);
      }



      /// <summary>
      /// Sends an ASCII character to the LCD. This can be used to read or write in the case 
      /// of custom characters
      /// </summary>
      /// <param name="c"></param>
      private void Putc(byte c)
      {
         DB7.Write((c & 0x80) != 0);
         DB6.Write((c & 0x40) != 0);
         DB5.Write((c & 0x20) != 0);
         DB4.Write((c & 0x10) != 0);
         Enable.Write(true); Enable.Write(false); //Toggle the Enable Pin

         DB7.Write((c & 0x08) != 0);
         DB6.Write((c & 0x04) != 0);
         DB5.Write((c & 0x02) != 0);
         DB4.Write((c & 0x01) != 0);
         Enable.Write(true); Enable.Write(false); //Toggle the Enable Pin
         //Thread.Sleep(1);
      }


      /// <summary>
      /// Sends a command to the LCD 
      /// </summary>
      /// <param name="c"></param>
      private void SendCmd(byte c)
      {
         RegisterSelect.Write(false); //set LCD to data mode

         DB7.Write((c & 0x80) != 0);
         DB6.Write((c & 0x40) != 0);
         DB5.Write((c & 0x20) != 0);
         DB4.Write((c & 0x10) != 0);
         Enable.Write(true); Enable.Write(false); //Toggle the Enable Pin

         DB7.Write((c & 0x08) != 0);
         DB6.Write((c & 0x04) != 0);
         DB5.Write((c & 0x02) != 0);
         DB4.Write((c & 0x01) != 0);
         Enable.Write(true); Enable.Write(false); //Toggle the Enable Pin
         Thread.Sleep(1);
         RegisterSelect.Write(true); //set LCD to data mode
      }
   }
}
