/*
Using a sensor like http://www.elecrow.com/sensors-c-111/proximity-c-111_113/hcsr04-ultrasonic-ranging-sensor-p-316.html
  
http://www.elecrow.com/download/HC_SR04%20Datasheet.pdf
 
(C) April 2014 devMobile.co.nz and CodeClub.org.nz
 
Inspired by 
 
https://github.com/coolkev/Netduino/blob/master/MotorShield/UltraSonicSensor.cs

*/
using System;
using System.Threading;
using Microsoft.SPOT;
using Microsoft.SPOT.Hardware;


namespace CodeClub.Hardware.Sensors
{
   class UltraSonicRanger : IDisposable
   {
      private readonly OutputPort triggerOutput;
      private readonly InterruptPort echoInterrupt;
      private long pulseStartTicks;
      private long pulseWidthTicks;
      private ManualResetEvent measurementComplete = new ManualResetEvent(false);

      
      
      public UltraSonicRanger(Cpu.Pin triggerPin, Cpu.Pin echoPin)
      {
         #region Diagnostic Assertions
         Debug.Assert(echoPin != Cpu.Pin.GPIO_NONE);
         Debug.Assert(triggerPin != Cpu.Pin.GPIO_NONE);
         Debug.Assert(echoPin != triggerPin);
         #endregion

         triggerOutput = new OutputPort(triggerPin, false);

         echoInterrupt = new InterruptPort(echoPin, true, Port.ResistorMode.Disabled, Port.InterruptMode.InterruptEdgeBoth);
         echoInterrupt.OnInterrupt += new NativeEventHandler(echoInterruptPort_OnInterrupt);
         echoInterrupt.DisableInterrupt();
      }

      
      
      void echoInterruptPort_OnInterrupt(uint data1, uint data2, DateTime time)
      {
         if (data2 == 0) // falling edge, end of pulse
         {
            pulseWidthTicks = time.Ticks - pulseStartTicks;
            measurementComplete.Set();
         }
         else
         {
            pulseStartTicks = time.Ticks;
         }
      }

      
      
      public int TakeReading()
      {
         measurementComplete.Reset();

         echoInterrupt.EnableInterrupt();
         triggerOutput.Write(false);
         Thread.Sleep(2);
         triggerOutput.Write(true);
         Thread.Sleep(10);
         triggerOutput.Write(false);
         Thread.Sleep(2);

         if (!measurementComplete.WaitOne(100, false))
         {
            throw new ApplicationException("No echo within timeout distance measurement failed") ;
         }

         return (int)(pulseWidthTicks / 58);
      }

      
      
      public void Dispose()
      {
         echoInterrupt.Dispose();
         triggerOutput.Dispose();
      }
   }
}
