﻿/*
 * Copyright 2013 Bryn Lewis (http://www.devmobile.co.nz/)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
using System;
using System.Diagnostics;
using System.Net;
using System.Security.Cryptography.X509Certificates;
using System.Threading;

using Microsoft.SPOT;
using Gadgeteer;
using GHI.Premium.Net;



namespace AzureServiceBusClient
{
   public partial class Program
   {
      private static AutoResetEvent ipAddressReady = new AutoResetEvent(false);
      private X509Certificate[] caCerts;


      
      // This method is run when the mainboard is powered up or reset.   
      void ProgramStarted()
      {
         // Use Debug.Print to show messages in Visual Studio's "Output" window during debugging.
         Debug.Print("Program Started");

         Debug.EnableGCMessages(false);

         Microsoft.SPOT.Hardware.Utility.SetLocalTime(new DateTime(2013, 3, 1));

         ethernet_J11D.Interface.Open();

         if (!ethernet_J11D.Interface.IsActivated)
         {
            NetworkInterfaceExtension.AssignNetworkingStackTo(ethernet_J11D.Interface);
         }
         ethernet_J11D.Interface.NetworkAddressChanged += new NetworkInterfaceExtension.NetworkAddressChangedEventHandler(NetworkAddressChanged);
         ethernet_J11D.Interface.CableConnectivityChanged += new EthernetBuiltIn.CableConnectivityChangedEventHandler(CableConnectivityChanged);

         // The code below (Dynamic configuration or static configuration) could be removed once the MFDeploy fixed.
         // This is a known issue see (http://wiki.tinyclr.com/index.php?title=NETMF4.2_developer)

         // For networks with dynamic configuration
         ethernet_J11D.Interface.NetworkInterface.EnableDhcp();
         ethernet_J11D.Interface.NetworkInterface.EnableDynamicDns();

         // For networks with static IP
         //Ethernet.NetworkInterface.EnableStaticIP("x.x.x.x", "y.y.y.y", "z.z.z.z");
         //Ethernet.NetworkInterface.EnableStaticDns(new[] { "x.x.x.x", "y.y.y.y" });

         ipAddressReady.WaitOne();

         // Display network config for debugging
         Debug.Print("Network configuration");
         Debug.Print(" Network interface type: " + ethernet_J11D.Interface.NetworkInterface.NetworkInterfaceType.ToString());
         Debug.Print(" MAC Address: " + BytesToHexString(ethernet_J11D.Interface.NetworkInterface.PhysicalAddress));
         Debug.Print(" DHCP enabled: " + ethernet_J11D.Interface.NetworkInterface.IsDhcpEnabled.ToString());
         Debug.Print(" Dynamic DNS enabled: " + ethernet_J11D.Interface.NetworkInterface.IsDynamicDnsEnabled.ToString());
         Debug.Print(" IP Address: " + ethernet_J11D.Interface.NetworkInterface.IPAddress.ToString());
         Debug.Print(" Subnet Mask: " + ethernet_J11D.Interface.NetworkInterface.SubnetMask.ToString());
         Debug.Print(" Gateway: " + ethernet_J11D.Interface.NetworkInterface.GatewayAddress.ToString());
         foreach (string dnsAddress in ethernet_J11D.Interface.NetworkInterface.DnsAddresses)
         {
            Debug.Print(" DNS Server: " + dnsAddress.ToString());
         }

         byte[] ca = Resources.GetBytes(Resources.BinaryResources.GTERoot );
         caCerts = new X509Certificate[] { new X509Certificate(ca) };

         // Setup a timer to call google every so often
         Gadgeteer.Timer timer = new Gadgeteer.Timer(10000);
         timer.Tick += new Gadgeteer.Timer.TickEventHandler(Timer_Tick);
         timer.Start();
      }


      void CableConnectivityChanged(object sender, EventArgs e)
      {
         Debug.Print("CableConnectivityChanged");
         Debug.Print(" Cable connected: " + ethernet_J11D.Interface.IsCableConnected.ToString());

         if (ethernet_J11D.Interface.NetworkInterface.IPAddress != IPAddress.Any.ToString())
         {
            ipAddressReady.Set();
         }
      }



      void NetworkAddressChanged(object sender, EventArgs e)
      {
         Debug.Print("NetworkAddressChanged");
         Debug.Print(" IP Address: " + ethernet_J11D.Interface.NetworkInterface.IPAddress.ToString());

         if (ethernet_J11D.Interface.NetworkInterface.IPAddress != IPAddress.Any.ToString())
         {
            ipAddressReady.Set();
         }
      }

      
      
      public static string BytesToHexString(byte[] bytes)
      {
         string hexString = string.Empty;

         // Create a character array for hexidecimal conversion.
         const string hexChars = "0123456789ABCDEF";

         // Loop through the bytes.
         for (byte b = 0; b < bytes.Length; b++)
         {
            if (b > 0)
               hexString += "-";

            // Grab the top 4 bits and append the hex equivalent to the return string.        
            hexString += hexChars[bytes[b] >> 4];

            // Mask off the upper 4 bits to get the rest of it.
            hexString += hexChars[bytes[b] & 0x0F];
         }

         return hexString;
      }


      private static Boolean alreadyRunning = false ;



      void Timer_Tick(Gadgeteer.Timer timer)
      {
         Debug.Print("Timer_Tick");

         if (alreadyRunning)
         {
            return;
         }
         alreadyRunning = true;

         try
         {
            using (HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(@"https://kdgthesis.servicebus.windows.net/robot0/messages/head?timeout=60"))
            {
               request.Method = "GET";
               request.HttpsAuthentCerts = caCerts;

               // Enable these options to suit your environment
               //request.Proxy = new WebProxy("myproxy.myorganisation.com", true);
               //request.Credentials = new NetworkCredential("myusername", "mytopsecretpassword"); 

               using (HttpWebResponse response = (HttpWebResponse)request.GetResponse())
               {
                  if (response.StatusCode == HttpStatusCode.OK)
                  {
                     PulseDebugLED();
                  }
                  Debug.Print(response.StatusCode + " : " + response.StatusDescription);
               }
            }
         }
         catch (WebException we)
         {
            Debug.Print(we.Message);
         }
         finally
         {
            alreadyRunning = false;
         }
      }
   }
}
